
/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#ifndef MBED_FILESYSTEM_H
#define MBED_FILESYSTEM_H

#include "platform/platform.h"

#include "platform/FileBase.h"
#include "platform/FileHandle.h"
#include "platform/DirHandle.h"
#include "platform/FileSystemLike.h"
#include "BlockDevice.h"

namespace mbed {
/** \addtogroup filesystem */
/** @{*/


// Opaque pointer representing files and directories
typedef void *fs_file_t;
typedef void *fs_dir_t;

// Predeclared classes
class Dir;
class mFile;

/** A filesystem object provides filesystem operations and file operations
 *  for the File and Dir classes on a block device.
 *
 *  Implementations must provide at minimum file operations and mount
 *  operations for block devices.
 *
 * @note Synchronization level: Set by subclass
 */
class FileSystem : public FileSystemLike {
public:

    /** FileSystem lifetime
     *
     *  @param name       Name to add filesystem to tree as
     */
    FileSystem(const char *name = NULL);
    virtual ~FileSystem() {}

    /** Return the default filesystem
     *
     * Returns the default FileSystem base on the default BlockDevice configuration.
     * Use the components in target.json or application config to change
     * the default block device and affect the default filesystem.
     * SD block device => FAT filesystem
     * SPIF block device => LITTLE filesystem
     * DATAFLASH block device => LITTLE filesystem
     *
     * An application can override all target settings by implementing
     * FileSystem::get_default_instance() themselves - the default
     * definition is weak, and calls get_target_default_instance().
     */
    static FileSystem *get_default_instance();

    /** Mounts a filesystem to a block device
     *
     *  @param bd       BlockDevice to mount to
     *  @return         0 on success, negative error code on failure
     */
    virtual int mount(BlockDevice *bd) = 0;

    /** Unmounts a filesystem from the underlying block device
     *
     *  @return         0 on success, negative error code on failure
     */
    virtual int unmount() = 0;

    /** Reformats a filesystem, results in an empty and mounted filesystem
     *
     *  @param bd       BlockDevice to reformat and mount. If NULL, the mounted
     *                  block device will be used.
     *                  Note: if mount fails, bd must be provided.
     *                  Default: NULL
     *  @return         0 on success, negative error code on failure
     */
    virtual int reformat(BlockDevice *bd = NULL);

    /** Remove a file from the filesystem.
     *
     *  @param path     The name of the file to remove.
     *  @return         0 on success, negative error code on failure
     */
    virtual int remove(const char *path);

    /** Rename a file in the filesystem.
     *
     *  @param path     The name of the file to rename.
     *  @param newpath  The name to rename it to
     *  @return         0 on success, negative error code on failure
     */
    virtual int rename(const char *path, const char *newpath);

    /** Store information about the file in a stat structure
     *
     *  @param path     The name of the file to find information about
     *  @param st       The stat buffer to write to
     *  @return         0 on success, negative error code on failure
     */
    virtual int stat(const char *path, struct stat *st);

    /** Create a directory in the filesystem.
     *
     *  @param path     The name of the directory to create.
     *  @param mode     The permissions with which to create the directory
     *  @return         0 on success, negative error code on failure
     */
    virtual int mkdir(const char *path, mode_t mode);

    /** Store information about the mounted filesystem in a statvfs structure
     *
     *  @param path     The name of the file to find information about
     *  @param buf      The stat buffer to write to
     *  @return         0 on success, negative error code on failure
     */
     virtual int statvfs(const char *path, struct statvfs *buf);

protected:
    friend class mFile;
    friend class Dir;

    /** Open a file on the filesystem
     *
     *  @param file     Destination for the handle to a newly created file
     *  @param path     The name of the file to open
     *  @param flags    The flags to open the file in, one of O_RDONLY, O_WRONLY, O_RDWR,
     *                  bitwise or'd with one of O_CREAT, O_TRUNC, O_APPEND
     *  @return         0 on success, negative error code on failure
     */
    virtual int file_open(fs_file_t *file, const char *path, int flags) = 0;

    /** Close a file
     *
     *  @param file     File handle
     *  @return         0 on success, negative error code on failure
     */
    virtual int file_close(fs_file_t file) = 0;

    /** Read the contents of a file into a buffer
     *
     *  @param file     File handle
     *  @param buffer   The buffer to read in to
     *  @param size     The number of bytes to read
     *  @return         The number of bytes read, 0 at end of file, negative error on failure
     */
    virtual ssize_t file_read(fs_file_t file, void *buffer, size_t size) = 0;

    /** Write the contents of a buffer to a file
     *
     *  @param file     File handle
     *  @param buffer   The buffer to write from
     *  @param size     The number of bytes to write
     *  @return         The number of bytes written, negative error on failure
     */
    virtual ssize_t file_write(fs_file_t file, const void *buffer, size_t size) = 0;

    /** Flush any buffers associated with the file
     *
     *  @param file     File handle
     *  @return         0 on success, negative error code on failure
     */
    virtual int file_sync(fs_file_t file);

    /** Check if the file in an interactive terminal device
     *  If so, line buffered behaviour is used by default
     *
     *  @param file     File handle
     *  @return         True if the file is a terminal
     */
    virtual int file_isatty(fs_file_t file);

    /** Move the file position to a given offset from from a given location
     *
     *  @param file     File handle
     *  @param offset   The offset from whence to move to
     *  @param whence   The start of where to seek
     *      SEEK_SET to start from beginning of file,
     *      SEEK_CUR to start from current position in file,
     *      SEEK_END to start from end of file
     *  @return         The new offset of the file
     */
    virtual off_t file_seek(fs_file_t file, off_t offset, int whence) = 0;

    /** Get the file position of the file
     *
     *  @param file     File handle
     *  @return         The current offset in the file
     */
    virtual off_t file_tell(fs_file_t file);

    /** Rewind the file position to the beginning of the file
     *
     *  @param file     File handle
     *  @note This is equivalent to file_seek(file, 0, FS_SEEK_SET)
     */
    virtual void file_rewind(fs_file_t file);

    /** Get the size of the file
     *
     *  @param file     File handle
     *  @return         Size of the file in bytes
     */
    virtual off_t file_size(fs_file_t file);

    /** Open a directory on the filesystem
     *
     *  @param dir      Destination for the handle to the directory
     *  @param path     Name of the directory to open
     *  @return         0 on success, negative error code on failure
     */
    virtual int dir_open(fs_dir_t *dir, const char *path);

    /** Close a directory
     *
     *  @param dir      Dir handle
     *  @return         0 on success, negative error code on failure
     */
    virtual int dir_close(fs_dir_t dir);

    /** Read the next directory entry
     *
     *  @param dir      Dir handle
     *  @param ent      The directory entry to fill out
     *  @return         1 on reading a filename, 0 at end of directory, negative error on failure
     */
    virtual ssize_t dir_read(fs_dir_t dir, struct dirent *ent);

    /** Set the current position of the directory
     *
     *  @param dir      Dir handle
     *  @param offset   Offset of the location to seek to,
     *                  must be a value returned from dir_tell
     */
    virtual void dir_seek(fs_dir_t dir, off_t offset);

    /** Get the current position of the directory
     *
     *  @param dir      Dir handle
     *  @return         Position of the directory that can be passed to dir_rewind
     */
    virtual off_t dir_tell(fs_dir_t dir);

    /** Rewind the current position to the beginning of the directory
     *
     *  @param dir      Dir handle
     */
    virtual void dir_rewind(fs_dir_t dir);

    /** Get the sizeof the directory
     *
     *  @param dir      Dir handle
     *  @return         Number of files in the directory
     */
    virtual size_t dir_size(fs_dir_t dir);

protected:
    // Hooks for FileSystemHandle
    virtual int open(FileHandle **file, const char *path, int flags);
    virtual int open(DirHandle **dir, const char *path);
};


/** @}*/
} // namespace mbed

#endif
