/*
  HardwareSerial.cpp - Hardware serial library for Wiring
  Copyright (c) 2006 Nicholas Zambetti.  All right reserved.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
  
  Modified 23 November 2006 by David A. Mellis
  Modified 28 September 2010 by Mark Sproul
  Modified 14 August 2012 by Alarus
  Modified 3 December 2013 by Matthijs Kooijman
  Modified 22 June 2016 by Nozomu Fujita
  Modified 30 Aug 2016 by Yuuki Okamiya
*/

#include "HardwareSerial.h"
#include "HardwareSerial_private.h"
#include "serial_api.h"

#include <Arduino.h>

// Public Methods //////////////////////////////////////////////////////////////

void HardwareSerial::begin(unsigned long baud, byte config)
{
    _rx_buffer_head = 0;
    _rx_buffer_tail = 0;
    serial_init(&_serial, _tx, _rx);
    serial_baud(&_serial, baud);

    int bits = 8;
    SerialParity parity = ParityNone;
    int stop_bits = 1;
    const uint8_t SERIAL_DM = 0b00000111;
    const uint8_t SERIAL_D8 = 0b00000110;
    const uint8_t SERIAL_D7 = 0b00000100;
#if 0
    const uint8_t SERIAL_D6 = 0b00000010;
    const uint8_t SERIAL_D5 = 0b00000000;
#endif
    const uint8_t SERIAL_PM = 0b00110000;
    const uint8_t SERIAL_PN = 0b00000000;
    const uint8_t SERIAL_PE = 0b00100000;
    const uint8_t SERIAL_PO = 0b00110000;
    const uint8_t SERIAL_SM = 0b00001000;
    const uint8_t SERIAL_S1 = 0b00000000;
    const uint8_t SERIAL_S2 = 0b00001000;

    switch (config & SERIAL_DM) {
    case SERIAL_D8:
        bits = 8;
        break;
    case SERIAL_D7:
        bits = 7;
        break;
#if 0
    case SERIAL_D6:
        bits = 6;
        break;
    case SERIAL_D5:
        bits = 5;
        break;
#endif
    default:
        break;
    }
    switch (config & SERIAL_PM) {
    case SERIAL_PN:
        parity = ParityNone;
        break;
    case SERIAL_PE:
        parity = ParityEven;
        break;
    case SERIAL_PO:
        parity = ParityOdd;
        break;
    default:
        break;
    }
    switch (config & SERIAL_SM) {
    case SERIAL_S1:
        stop_bits = 1;
        break;
    case SERIAL_S2:
        stop_bits = 2;
        break;
    default:
        break;
    }
    serial_format(&_serial, bits, parity, stop_bits);

    serial_irq_set(&_serial, RxIrq, 0);
    serial_irq_set(&_serial, TxIrq, 0);
    serial_irq_handler(&_serial, HardwareSerial::_irq_handler, _id);
    serial_irq_set(&_serial, RxIrq, 1);
}

void HardwareSerial::end()
{
    flush();
    serial_free(&_serial);

    serial_irq_set(&_serial, RxIrq, 0);
    serial_irq_handler(&_serial, 0, _id);
}

int HardwareSerial::available(void)
{
    return int((SERIAL_RX_BUFFER_SIZE + _rx_buffer_head - _rx_buffer_tail) % SERIAL_RX_BUFFER_SIZE);
}

int HardwareSerial::peek(void)
{
    int c = -1;
    // if the head isn't ahead of the tail, we don't have any characters
    if (available() > 0) {
        c = _rx_buffer[_rx_buffer_tail];
    }
    return c;
}

int HardwareSerial::read(void)
{
    int c = peek();
    if (c >= 0) {
        _rx_buffer_tail = (rx_buffer_index_t)((_rx_buffer_tail + 1) % SERIAL_RX_BUFFER_SIZE);
    }
    return c;
}

void HardwareSerial::flush()
{
    while (!serial_writable(&_serial)) {
        ;
    }
}

void HardwareSerial::_irq_handler(uint32_t id, SerialIrq irq_type)
{
    if (irq_type == RxIrq) {
        switch (id) {
        case 0:
            Serial0._rx_complete_irq();
            break;
        case 1:
            Serial1._rx_complete_irq();
            break;
        case 2:
            Serial2._rx_complete_irq();
            break;
        case 3:
            Serial3._rx_complete_irq();
            break;
        case 4:
            Serial4._rx_complete_irq();
            break;
        case 5:
            Serial5._rx_complete_irq();
            break;
        case 6:
            Serial6._rx_complete_irq();
            break;
        case 7:
            Serial7._rx_complete_irq();
            break;
        case 8:
            Serial._rx_complete_irq();
            break;
        default:
            break;
        }
    }
}

size_t HardwareSerial::write(uint8_t c)
{
    serial_putc(&_serial, c);
    return 1;
}

HardwareSerial Serial(8, USBTX, USBRX);
HardwareSerial Serial0(0, P2_14, P2_15);
HardwareSerial Serial1(1, P2_5, P2_6);
HardwareSerial Serial2(2, P6_3, P6_2);
HardwareSerial Serial3(3, P5_3, P5_4);
#if 1
HardwareSerial Serial4(4, P8_14, P8_15);
#else
HardwareSerial Serial4(4, P5_0, P5_1);
#endif
#if 1
HardwareSerial Serial5(5, P8_13, P8_11);
#else
HardwareSerial Serial5(5, P6_6, P6_7);
#endif
HardwareSerial Serial6(6, P5_6, P5_7);
HardwareSerial Serial7(7, P7_4, P7_5);
