// Copyright 2018 Proyectos y Sistemas de Mantenimiento SL (eProsima).
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#ifdef __RTOS
#include "platform.h"
#endif

#if (BSP_CFG_RTOS_USED != 1)
#include <fcntl.h>
#endif
#include <stdio.h>
#include <errno.h>

#include "micrortps_serial_transport.h"

static micrortps_locator_t* g_serial_locators[MAX_NUM_LOCATORS] = {0};
static uint8_t g_num_locators = 0;

/* This Demo use one serial channel.
 * If use multiple serial channel, g_poll_fds should be used. */
#if (BSP_CFG_RTOS_USED != 1)
#ifndef _WIN32

static struct pollfd g_poll_fds[MAX_NUM_LOCATORS] = {0};

#endif // _WIN32
#endif // BSP_CFG_RTOS_USED != 1

/// extern function declarations

uint16_t crc16_byte(uint16_t crc, const uint8_t data);
uint16_t crc16(const uint8_t* buffer, size_t len);
int extract_message(octet_t* out_buffer, const size_t buffer_len, buffer_t* internal_buffer);


/// local function declarations

locator_id_t         create_serial_locator (const char* device, locator_id_t locator_id, micrortps_locator_t* const locator);
int                  remove_serial_locator (const locator_id_t locator_id);
#if (BSP_CFG_RTOS_USED == 1)
sci_hdl_t            open_serial_locator   (micrortps_locator_t* const locator);
#else
int                  open_serial_locator   (micrortps_locator_t* const locator);
#endif // BSP_CFG_RTOS_USED == 1
int                  close_serial_locator  (micrortps_locator_t* const locator);
int                  write_serial          (const void* buffer, const size_t len, micrortps_locator_t* const locator);
int                  read_serial           (micrortps_locator_t* const channel);
int                  send_serial           (const header_t* header, const octet_t* in_buffer, const size_t length, const locator_id_t locator_id);
int                  receive_serial        (octet_t* out_buffer, const size_t buffer_len, const locator_id_t locator_id, const uint16_t timeout_ms);
micrortps_locator_t* get_serial_locator    (const locator_id_t locator_id);


/// function definition

micrortps_locator_t* get_serial_locator(const locator_id_t locator_id)
{
    micrortps_locator_t* ret = NULL;

#ifdef _WIN32

    (void) locator_id;
    return ret;

#else

    for (int i = 0; i < g_num_locators; ++i)
    {
        if (NULL != g_serial_locators[i] &&
            g_serial_locators[i]->locator_id == locator_id)
        {
            ret = g_serial_locators[i];
            break;
        }
    }

    return ret;

#endif
}


locator_id_t create_serial_locator(const char* device, locator_id_t loc_id, micrortps_locator_t* const locator)
{

#ifdef _WIN32

    (void) device;
    (void) loc_id;
    (void) locator;
    return MICRORTPS_TRANSPORT_ERROR;

#else

    if (NULL == device || 0 == strlen(device) || UART_NAME_MAX_LENGTH <= strlen(device) ||
        NULL == locator || MAX_NUM_LOCATORS <= g_num_locators)
    {
        MICRORTPS_TRANSPORT_PRINTF("# create_serial_locator() -> BAD PARAMETERS!\n");
        return MICRORTPS_TRANSPORT_ERROR;
    }


    /// Fill locator struct

    memset(locator, 0, sizeof(micrortps_locator_t));
    locator->rx_buffer.buff_pos = 0;
    locator->open = false;
    locator->poll_ms = DFLT_POLL_MS;
    locator->channel.kind = LOC_SERIAL;
    serial_channel_t* channel = &locator->channel._.serial;
    memcpy(channel->uart_name, device, UART_NAME_MAX_LENGTH);
#if (BSP_CFG_RTOS_USED == 1)
    channel->uart_fd = NULL;
#else
    channel->uart_fd = -1;
#endif // BSP_CFG_RTOS_USED == 1
    channel->baudrate = DFLT_BAUDRATE;

    locator->locator_id = -1;
    for (int i = 0; i < MAX_NUM_LOCATORS; ++i)
    {
        if (NULL == g_serial_locators[i])
        {
            locator->locator_id = loc_id;
            locator->idx = i;
            g_serial_locators[i] = locator;
            ++g_num_locators;
            break;
        }
    }

    #ifdef TRANSPORT_LOGS
    printf("> Create serial locator id: %d\n", locator->locator_id);
    #endif // TRANSPORT_LOGS

    return locator->locator_id;

#endif
}

int remove_serial_locator(const locator_id_t loc_id)
{
#ifdef _WIN32

    (void) loc_id;
    return MICRORTPS_TRANSPORT_ERROR;

#else

    micrortps_locator_t* locator = get_serial_locator(loc_id);
    if (NULL == locator)
    {
        return MICRORTPS_TRANSPORT_ERROR;
    }

    if (locator->open)
    {
        close_serial_locator(locator);
    }
    g_serial_locators[locator->idx] = NULL;

    return MICRORTPS_TRANSPORT_OK;

#endif
}

#if (BSP_CFG_RTOS_USED == 1)
sci_hdl_t open_serial_locator(micrortps_locator_t* const locator)
#else
int open_serial_locator(micrortps_locator_t* const locator)
#endif // BSP_CFG_RTOS_USED == 1
{

#ifdef _WIN32

    (void) locator;
    return MICRORTPS_TRANSPORT_ERROR;

#else

    if (NULL == locator)
    {
        return MICRORTPS_TRANSPORT_ERROR;
    }

    serial_channel_t* channel = &locator->channel._.serial;

#if (BSP_MCU_RX65N_2MB == 1)
    /* Initialize the USB serial port for the RSKRX65-2M. */
    sci_cfg_t   sci_config;
    sci_err_t   sci_err;

    /* Initialize the I/O port pins for communication on this SCI channel.
    * This is specific to the MCU and ports chosen. For the RSKRX65-2M we will use the
    * SCI channel connected to the USB serial port emulation. */
#if (BSP_CFG_BOARD_REVISION == 0)
    R_SCI_PinSet_SCI2();
#elif (BSP_CFG_BOARD_REVISION == 1)
    R_SCI_PinSet_SCI8();
#endif
    /* Set up the configuration data structure for asynchronous (UART) operation. */
    sci_config.async.baud_rate    = channel->baudrate;
    sci_config.async.clk_src      = SCI_CLK_INT;
    sci_config.async.data_size    = SCI_DATA_8BIT;
    sci_config.async.parity_en    = SCI_PARITY_OFF;
    sci_config.async.parity_type  = SCI_EVEN_PARITY;
    sci_config.async.stop_bits    = SCI_STOPBITS_1;
    sci_config.async.int_priority = 3; // 1=lowest, 15=highest

    /* OPEN ASYNC CHANNEL
    *  Provide address of the configure structure,
    *  the callback function to be assigned,
    *  and the location for the handle to be stored.*/
#if (BSP_CFG_BOARD_REVISION == 0)
    sci_err = R_SCI_Open(SCI_CH2, SCI_MODE_ASYNC, &sci_config, NULL, &channel->uart_fd);
#elif (BSP_CFG_BOARD_REVISION == 1)
    sci_err = R_SCI_Open(SCI_CH8, SCI_MODE_ASYNC, &sci_config, NULL, &channel->uart_fd);
#endif

    if (SCI_SUCCESS != sci_err)
    {
        MICRORTPS_TRANSPORT_PRINTF("failed to open serial device\n");
        return NULL; // return error
    }
#else
    // Open a serial port
    channel->uart_fd = open(channel->uart_name, O_RDWR | O_NOCTTY | O_NONBLOCK);
    if (channel->uart_fd < 0)
    {
        MICRORTPS_TRANSPORT_PRINTF_ARGS("failed to open device: %s (%d)\n", channel->uart_name, errno);
        return -errno;
    }

    // Try to set baud rate
    struct termios uart_config;
    int termios_state;

    // Back up the original uart configuration to restore it after exit
    if ((termios_state = tcgetattr(channel->uart_fd, &uart_config)) < 0)
    {
        int errno_bkp = errno;
        MICRORTPS_TRANSPORT_PRINTF_ARGS("ERR GET CONF %s: %d (%d)\n", channel->uart_name, termios_state, errno);
        close_serial_locator(locator);
        return -errno_bkp;
    }

    // Clear ONLCR flag (which appends a CR for every LF)
    cfsetospeed(&uart_config, (speed_t)channel->baudrate);
    cfsetispeed(&uart_config, (speed_t)channel->baudrate);

    uart_config.c_cflag |= (CLOCAL | CREAD);    /* ignore modem controls */
    uart_config.c_cflag &= ~CSIZE;
    uart_config.c_cflag |= CS8;         /* 8-bit characters */
    uart_config.c_cflag &= ~PARENB;     /* no parity bit */
    uart_config.c_cflag &= ~CSTOPB;     /* only need 1 stop bit */
    uart_config.c_cflag &= ~CRTSCTS;    /* no hardware flowcontrol */

    /* setup for non-canonical mode */
    uart_config.c_iflag &= ~(IGNBRK | BRKINT | PARMRK | ISTRIP | INLCR | IGNCR | ICRNL | IXON);
    uart_config.c_lflag &= ~(ECHO | ECHONL | ICANON | ISIG | IEXTEN);
    uart_config.c_oflag &= ~OPOST;

    /* fetch bytes as they become available */
    uart_config.c_cc[VMIN] = 1;
    uart_config.c_cc[VTIME] = 1;

    if ((termios_state = tcsetattr(channel->uart_fd, TCSANOW, &uart_config)) < 0)
    {
        int errno_bkp = errno;
        MICRORTPS_TRANSPORT_PRINTF_ARGS("ERR SET CONF %s (%d)\n", channel->uart_name, errno);
        close_serial_locator(locator);
        return -errno_bkp;
    }

    uint8_t aux_buffer[64];
    while (0 < read(channel->uart_fd, (void *)&aux_buffer, 64)); //clear previous data
#endif // BSP_MCU_RX65N_2MB == 1
    locator->open = true;
#if (BSP_CFG_RTOS_USED != 1)
    g_poll_fds[locator->idx].fd = channel->uart_fd;
    g_poll_fds[locator->idx].events = POLLIN;
#endif
    return channel->uart_fd;

#endif
}

int close_serial_locator(micrortps_locator_t* const locator)
{

#ifdef _WIN32

    (void) locator;
    return MICRORTPS_TRANSPORT_ERROR;

#else

    if (NULL == locator)
    {
        MICRORTPS_TRANSPORT_PRINTF("# close_serial_locator() -> BAD PARAMETERS!\n");
        return MICRORTPS_TRANSPORT_ERROR;
    }

    serial_channel_t* channel = &locator->channel._.serial;
#if (BSP_CFG_RTOS_USED == 1)
    if (SCI_SUCCESS != R_SCI_Close(channel->uart_fd))
#else
    if (0 != close(channel->uart_fd))
#endif // BSP_CFG_RTOS_USED == 1
    {
        MICRORTPS_TRANSPORT_PRINTF("# close_serial_locator() -> close error\n");
        return MICRORTPS_TRANSPORT_ERROR;
    }
#if (BSP_CFG_RTOS_USED == 1)
    channel->uart_fd = NULL;
#else
    channel->uart_fd = -1;
#endif // BSP_CFG_RTOS_USED == 1
    locator->open = false;
#if (BSP_CFG_RTOS_USED != 1)
    memset(&g_poll_fds[locator->idx], 0, sizeof(struct pollfd));
#endif
    #ifdef TRANSPORT_LOGS
    printf("> Close UART %s\n", locator->channel._.serial.uart_name);
    #endif // TRANSPORT_LOGS

    return MICRORTPS_TRANSPORT_OK;

#endif
}

int read_serial(micrortps_locator_t* const locator)
{

#ifdef _WIN32

    (void) locator;
    return MICRORTPS_TRANSPORT_ERROR;

#else

#if (BSP_CFG_RTOS_USED == 1)
    if (NULL == locator ||
            (!locator->open && NULL== open_serial_locator(locator)))
#else
    if (NULL == locator ||
            (!locator->open && 0 > open_serial_locator(locator)))
#endif // BSP_CFG_RTOS_USED == 1
    {
        MICRORTPS_TRANSPORT_PRINTF("# read_serial() -> BAD PARAMETERS!\n");
        return MICRORTPS_TRANSPORT_ERROR;
    }

    // TODO: for several channels this can be optimized
    int ret = 0;
#if (BSP_CFG_RTOS_USED == 1)
    uint16_t cnt;
    uint16_t time;
    time = locator->poll_ms;
    do
    {
        /* Get the number of bytes available in receive queue. */
        R_SCI_Control(locator->channel._.serial.uart_fd, SCI_CMD_RX_Q_BYTES_AVAIL_TO_READ, (void *)&cnt);
        if (cnt > 0)
        {
            break; // recieved data
        }
        time--;
        ms_sleep(1);
    } while (time > 0); // loop until timeout

    if (SCI_SUCCESS == R_SCI_Receive(locator->channel._.serial.uart_fd,
                   (uint8_t *) (locator->rx_buffer.buffer + locator->rx_buffer.buff_pos), cnt))
    {
        ret = cnt;
    }
#else
    int r = poll(g_poll_fds, g_num_locators, locator->poll_ms);
    if (r > 0 && (g_poll_fds[locator->idx].revents & POLLIN))
    {
        ret = read(locator->channel._.serial.uart_fd,
                   (void *) (locator->rx_buffer.buffer + locator->rx_buffer.buff_pos), // buffer pos
                   sizeof(locator->rx_buffer.buffer) - locator->rx_buffer.buff_pos);   // len
    }
#endif // BSP_CFG_RTOS_USED == 1
    return ret;

#endif
}


int receive_serial(octet_t* out_buffer, const size_t buffer_len, const locator_id_t locator_id, const uint16_t timeout_ms)
{

#ifdef _WIN32

    (void) out_buffer;
    (void) buffer_len;
    (void) locator_id;
    (void) timeout_ms;
    return MICRORTPS_TRANSPORT_ERROR;

#else

    if (NULL == out_buffer)
    {
        MICRORTPS_TRANSPORT_PRINTF("# receive_serial() -> BAD PARAMETERS!\n");
        return MICRORTPS_TRANSPORT_ERROR;
    }

    micrortps_locator_t* locator = get_serial_locator(locator_id);
#if (BSP_CFG_RTOS_USED == 1)
    if (NULL == locator ||
    	    (!locator->open && NULL == open_serial_locator(locator)))
#else
    if (NULL == locator ||
    	    (!locator->open && 0 > open_serial_locator(locator)))
#endif // BSP_CFG_RTOS_USED == 1
    {
        MICRORTPS_TRANSPORT_PRINTF("# receive_serial() -> error, serial not open\n");
        return MICRORTPS_TRANSPORT_ERROR;
    }

    locator->poll_ms = timeout_ms;

    int len = read_serial(locator);
    if (len <= 0)
    {
        int errsv = errno;

        if (errsv && EAGAIN != errsv && ETIMEDOUT != errsv)
        {
//            MICRORTPS_TRANSPORT_PRINTF("Read fail %d\n", errsv);
        }
    }
    else
    {
        // We read some bytes, trying extract a whole message
        locator->rx_buffer.buff_pos += len;
    }

    return extract_message(out_buffer, buffer_len, &locator->rx_buffer);

#endif
}

int write_serial(const void* buffer, const size_t len, micrortps_locator_t* const locator)
{

#ifdef _WIN32

    (void) buffer;
    (void) len;
    (void) locator;
    return MICRORTPS_TRANSPORT_ERROR;

#else
#if (BSP_CFG_RTOS_USED == 1)
    if (NULL == buffer       ||
        NULL == locator      ||
        (!locator->open && NULL == open_serial_locator(locator)))
#else
    if (NULL == buffer       ||
        NULL == locator      ||
        (!locator->open && 0 > open_serial_locator(locator)))
#endif // BSP_CFG_RTOS_USED == 1
#endif
    {
        MICRORTPS_TRANSPORT_PRINTF("# write_serial() -> BAD PARAMETERS!\n");
        return MICRORTPS_TRANSPORT_ERROR;
    }

#if (BSP_CFG_RTOS_USED == 1)
    sci_err_t sci_err;
    uint32_t retry = 0xFFFF;
    do
    {
        sci_err = R_SCI_Send(locator->channel._.serial.uart_fd, (uint8_t *)buffer, (uint16_t)len);
        retry--;
    } while ((SCI_ERR_XCVR_BUSY == sci_err) && (retry > 0)); // retry if previous transmission still in progress.

    if (SCI_SUCCESS == sci_err)
    {
        return (int)len;
    } else {
        return 0; // failed to send data
    }
#else
    return (int)write(locator->channel._.serial.uart_fd, buffer, len);
#endif // BSP_CFG_RTOS_USED == 1
}

int send_serial(const header_t* header, const octet_t* in_buffer, const size_t length, const locator_id_t locator_id)
{

#ifdef _WIN32

    (void) header;
    (void) in_buffer;
    (void) length;
    (void) locator_id;
    return MICRORTPS_TRANSPORT_ERROR;

#else

    if (NULL == in_buffer)
    {
        return MICRORTPS_TRANSPORT_ERROR;
    }

    micrortps_locator_t* locator = get_serial_locator(locator_id);
#if (BSP_CFG_RTOS_USED == 1)
    if (NULL == locator      ||
        (!locator->open && NULL == open_serial_locator(locator)))
#else
    if (NULL == locator      ||
        (!locator->open && 0 > open_serial_locator(locator)))
#endif // BSP_CFG_RTOS_USED == 1
    {
        MICRORTPS_TRANSPORT_PRINTF("# send_serial() -> error, serial not open\n");
        return MICRORTPS_TRANSPORT_ERROR;
    }

    int len = write_serial(header, sizeof(header_t), locator);
    if (len != sizeof(header_t))
    {
        return len;
    }

    len = write_serial(in_buffer, length, locator);

    return len; // only payload, + sizeof(header); for real size.

#endif
}
