/* QImode div/mod functions for the GCC support library for the Renesas RL78 processors.
   Copyright (C) 2017 Free Software Foundation, Inc.
   Contributed by Alex Panek.

   This file is part of GCC.

   GCC is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3, or (at your option)
   any later version.

   GCC is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   Under Section 7 of GPL version 3, you are granted additional
   permissions described in the GCC Runtime Library Exception, version
   3.1, as published by the Free Software Foundation.

   You should have received a copy of the GNU General Public License and
   a copy of the GCC Runtime Library Exception along with this program;
   see the files COPYING3 and COPYING.RUNTIME respectively.  If not, see
   <http://www.gnu.org/licenses/>.  */

#include "vregs.h"

#if defined __RL78_MUL_NONE__

; inputs:   A  = numerator   (unsigned 8 bit value)
;           X  = denominator (unsigned 8 bit value)
; output:   A  = quotient    (unsigned 8 bit value)
;           X  = remainder   (unsigned 8 bit value)
; clobbers: -

START_FUNC ___udivmodqi3

    push    bc
    xch     a, x               ; A = den, X = num
    movw    bc, #0x100         ; bit (B) = 1, quot (C) = 0
    cmp     x, a               ; cmp(den, num)
    bnh     $loop2_past_cmp    ; if (num <= den) goto loop2

loop1:
    cmp0    b                  ; cmp(bit, 0)
    bz      $end               ; if (bit == 0) goto end
    bt      a.7, $pre_loop2    ; if ((int)den<0) goto pre_loop2

    shl     a, 1               ; den <<= 1
    shl     b, 1               ; bit <<= 1

    cmp     a, x               ; cmp(num, den)
    bc      $loop1             ; if (den < num) goto loop1

pre_loop2:
    cmp0    b                  ; cmp(bit, 0)
    bz      $end               ; if (bit == 0) goto end

loop2:
    cmp     x, a               ; cmp(den, num)
loop2_past_cmp:
    bc      $skip              ; if (num < den) goto skip

    sub     x, a               ; num -= den
    xch     a, c
    or      a, b
    xch     a, c               ; quot |= bit

skip:
    shr     a, 1               ; den >>= 1
    xch     a, b
    shr     a, 1
    xch     a, b               ; bit >>= 1

    cmp0    b                  ; cmp(bit, 0)
    bnz     $loop2             ; if (bit != 0) goto loop2

end:
    mov     a, c               ; quotient is returned in A
    pop     bc                 ; remainder is returned in X
    ret                        ; (already there)

END_FUNC ___udivmodqi3

#elif defined  __RL78_MUL_G13__

; inputs:   A  = numerator   (unsigned 8 bit value)
;           X  = denominator (unsigned 8 bit value)
; output:   A  = quotient    (unsigned 8 bit value)
;           X  = remainder   (unsigned 8 bit value)
; clobbers: -

START_FUNC ___udivmodqi3

    mov    !MDUC, #0xC0        ; divison without interrupt generation
    push   ax                  ; save numerator

    clrb   a                   ; zero-extend denominator to 16 bits (AX)
    movw   MDBL, ax            ; set lower 16 bits of the divisor

    pop    ax
    shrw   ax, 8               ; zero-extend numerator to 16 bits (AX)
    movw   MDAL, ax            ; set lower 16 bits of the dividend

    clrw   ax
    movw   MDAH, ax            ; set/clear the upper 16 bits of the dividend
    movw   MDBH, ax            ; set/clear the upper 16 bits of the divisor

    mov    !MDUC, #0xC1        ; start the division operation on the peripheral
0:  mov    a, !MDUC            ; then wait for its completion (signalled by a
    bt     a.0, $0b            ; cleared DIVST bit)

    mov    a, MDAL             ; quotient is returned in A
    mov    x, !MDCL            ; remainder is returned in X
    ret

END_FUNC ___udivmodqi3

#elif defined  __RL78_MUL_G14__

; inputs:   A  = numerator   (unsigned 8 bit value)
;           X  = denominator (unsigned 8 bit value)
; output:   A  = quotient    (unsigned 8 bit value)
;           X  = remainder   (unsigned 8 bit value)
; clobbers: -

START_FUNC ___udivmodqi3

    push   de

    movw   de, ax
    shrw   ax, 8               ; zero-extend numerator to 16 bits (AX)
    mov    d, #0               ; zero-extend denominator to 16 bits (DE)

    divhu                      ; AX,DE <- AX : DE
    nop                        ; RENESAS TECHNICAL UPDATE TN-RL*-A025C/E

    mov    a, x                ; quotient is returned in A
    mov    x, r_4              ; remainder is returned in X

    pop    de
    ret

END_FUNC ___udivmodqi3

#else

#error "Unknown RL78 hardware multiply/divide support"

#endif

#if 0 /* DISABLED - NOT USED/USEABLE? */

; Signed QImode divisions (use ___udivmodqi3 as a primitive)

; inputs:   A  = numerator   (signed 8 bit value)
;           X  = denominator (signed 8 bit value)
; output:   A  = quotient    (signed 8 bit value)
; clobbers: C

START_FUNC ___divqi3

    clrb   c                       ; neg = 0
    bf     a.7, $0f

    xor    a, #255                 ; a = -a
    inc    a
    oneb   c                       ; neg = 1

0:
    bf     r_0.7, $1f

    xor    r_0, #255               ; b = -b
    inc    x
    xor    r_2, #1                 ; neg = !neg

1:
    call   $!___udivmodqi3

    cmp0   c                       ; if (!neg) return
    sknz
    ret
    
    xor    a, #255                 ; a = -a
    inc    a
    ret
    
END_FUNC ___divqi3

; inputs:   A  = numerator   (signed 8 bit value)
;           X  = denominator (signed 8 bit value)
; output:   A  = remainder   (signed 8 bit value)
; clobbers: C

START_FUNC ___modqi3

    clrb   c                       ; neg = 0
    bf     a.7, $0f

    xor    a, #255                 ; a = -a
    inc    a
    oneb   c                       ; neg = 1

0:
    bf     r_0.7, $1f

    xor    r_0, #255               ; b = -b
    inc    x

1:
    call   $!___udivmodqi3
    
    cmp0   c                       ; if (neg) goto 2
    bnz    $2f
    
    mov    a, x                    ; return in A reg
    ret

2:    
    clrb   a
    sub    a, x                    ; return A = -X
    ret
    
END_FUNC ___modqi3

#endif